/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, node: true, plusplus: true, devel: true, nomen: true, indent: 4, expr:true */
/*global beforeEach: false, afterEach: false, it: false, describe: false, expect: false*/

"use strict";

var sinon = require("sinon"),
    FileUtils = require("../lib/file-utils"),
    PreviewCache = require("../lib/preview-cache"),
    Q = require("q"),
    Buffer = require("buffer").Buffer;

describe("PreviewCache", function () {
    var sandbox = sinon.sandbox.create(),
        connectionId = 21,
        connectionId2 = 22,
        tempPath = "/tmp/foo.png",
        tempPath2 = "/tmp/bar.png",
        tempPath3 = "/tmp/batman.jpg",
        tempPath4 = "/tmp/robin.jpg",
        buffer = new Buffer([0,1,2,3,4,5,6,7]);

    beforeEach(function () {
        sandbox.stub(FileUtils, "removeFile").returns(Q.resolve());
    });

    afterEach(function () {
        sandbox.restore();
        PreviewCache.reset();
    });

    it("can insert and get temporary asset paths", function () {
        var previewId = PreviewCache.insert(connectionId, tempPath);
        expect(PreviewCache.get(previewId).tempAssetPath).to.equal(tempPath);
    });

    it("can set and retrieve buffer", function () {
        var previewId = PreviewCache.insert(connectionId);
        PreviewCache.setBuffer(previewId, buffer);

        var previewItem = PreviewCache.get(previewId);
        expect(previewItem).to.not.equal(undefined);
        expect(previewItem.connectionId).to.equal(connectionId);
        expect(previewItem.buffer).to.equal(buffer);
        expect(previewItem.tempAssetPath).to.equal(undefined);
    });

    it("can insert and get temporary asset paths associated with different connections", function () {
        var previewId = PreviewCache.insert(connectionId, tempPath),
            previewId2 = PreviewCache.insert(connectionId2, tempPath2);

        expect(PreviewCache.get(previewId).tempAssetPath).to.equal(tempPath);
        expect(PreviewCache.get(previewId2).tempAssetPath).to.equal(tempPath2);
    });

    it("returns undefined for a invalid preview id", function () {
        expect(PreviewCache.get(123)).to.equal(undefined);
    });

    it("can delete a preview by id", function () {
        var previewId = PreviewCache.insert(connectionId, tempPath);
        expect(PreviewCache.get(previewId).tempAssetPath).to.equal(tempPath);

        expect(PreviewCache.del(previewId)).to.equal(tempPath);
        expect(PreviewCache.get(previewId)).to.equal(undefined);
        expect(FileUtils.removeFile).to.be.calledOnce.and.to.be.calledWith(tempPath);
    });

    it("ignores a deletion request for an invalid preview id", function () {
        // We're mostly making sure this doesn't crash.
        expect(PreviewCache.del(567)).to.equal(undefined);
    });

    it("can delete all previews per connection", function () {
        var previewId = PreviewCache.insert(connectionId, tempPath),
            previewId2 = PreviewCache.insert(connectionId2, tempPath2),
            previewId3 = PreviewCache.insert(connectionId, tempPath3),
            previewId4 = PreviewCache.insert(connectionId2, tempPath4);

        expect(PreviewCache.get(previewId).tempAssetPath).to.equal(tempPath);
        expect(PreviewCache.get(previewId2).tempAssetPath).to.equal(tempPath2);
        expect(PreviewCache.get(previewId3).tempAssetPath).to.equal(tempPath3);
        expect(PreviewCache.get(previewId4).tempAssetPath).to.equal(tempPath4);

        expect(PreviewCache.deleteAllPreviewsForConnection(connectionId)).to.eql([tempPath, tempPath3]);
        expect(PreviewCache.get(previewId)).to.equal(undefined);
        expect(PreviewCache.get(previewId2).tempAssetPath).to.equal(tempPath2);
        expect(PreviewCache.get(previewId3)).to.equal(undefined);
        expect(PreviewCache.get(previewId4).tempAssetPath).to.equal(tempPath4);
        expect(FileUtils.removeFile.callCount).to.equal(2);
        expect(FileUtils.removeFile).to.be.calledWith(tempPath)
            .and.to.be.calledWith(tempPath3);

        expect(PreviewCache.deleteAllPreviewsForConnection(connectionId2)).to.eql([tempPath2, tempPath4]);
        expect(PreviewCache.get(previewId)).to.equal(undefined);
        expect(PreviewCache.get(previewId2)).to.equal(undefined);
        expect(PreviewCache.get(previewId3)).to.equal(undefined);
        expect(PreviewCache.get(previewId4)).to.equal(undefined);
        expect(FileUtils.removeFile.callCount).to.equal(4);
        expect(FileUtils.removeFile).to.be.calledWith(tempPath2)
            .and.to.be.calledWith(tempPath4);
    });

    it("deleteAllPreviewsForConnection() deletes buffer previews, but does not return an entry", function () {
        var previewId1 = PreviewCache.insert(connectionId),
            previewId2 = PreviewCache.insert(connectionId),
            previewId3 = PreviewCache.insert(connectionId);

        PreviewCache.setBuffer(previewId1, buffer);
        PreviewCache.setBuffer(previewId2, buffer);
        PreviewCache.setBuffer(previewId3, buffer);

        expect(Object.keys(PreviewCache._cache).length).to.be.equal(3);
        expect(PreviewCache.deleteAllPreviewsForConnection(connectionId)).to.eql([]);
        expect(Object.keys(PreviewCache._cache).length).to.be.equal(0);
    });

    it("ignores a preview deletion request for a connection id with no previews", function () {
        // No particular assertions. Let's just make sure this doesn't crash.
        PreviewCache.deleteAllPreviewsForConnection(connectionId);
    });

    it("can reset when non-empty", function () {
        var previewId = PreviewCache.insert(connectionId, tempPath),
            previewId2 = PreviewCache.insert(connectionId2, tempPath2),
            previewId3 = PreviewCache.insert(connectionId, undefined);

        PreviewCache.setBuffer(previewId3, buffer);

        expect(PreviewCache.get(previewId).tempAssetPath).to.equal(tempPath);
        expect(PreviewCache.get(previewId2).tempAssetPath).to.equal(tempPath2);
        expect(PreviewCache.get(previewId3).buffer).to.equal(buffer);

        PreviewCache.reset();
        expect(PreviewCache.get(previewId)).to.equal(undefined);
        expect(PreviewCache.get(previewId2)).to.equal(undefined);
        expect(PreviewCache.get(previewId3)).to.equal(undefined);
    });

    it("can reset when empty", function () {
        expect(PreviewCache.reset.bind(PreviewCache)).to.not.throw(Error);
    });
});
